package gov.cms.grouper.snf.component.v100.logic;

import static gov.cms.grouper.snf.component.v100.TestUtil.of;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.spy;

import gov.cms.grouper.snf.SnfTables;
import gov.cms.grouper.snf.component.v100.logic.NtaLogic.ParenteralIvFeeding;
import gov.cms.grouper.snf.lego.SnfUtils;
import gov.cms.grouper.snf.model.Assessment;
import gov.cms.grouper.snf.model.SnfDiagnosisCode;
import gov.cms.grouper.snf.model.table.NtaCmgRow;
import gov.cms.grouper.snf.model.table.NtaComorbidityRow;
import gov.cms.grouper.snf.util.ClaimInfo;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.SortedSet;
import java.util.function.BiFunction;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class NtaLogicTest {

  public final static int version = 100;

  public static NtaLogic make() {
    return make(Arrays.asList(), Arrays.asList());
  }

  public static NtaLogic make(List<Assessment> assessments, List<SnfDiagnosisCode> secondary) {
    ClaimInfo claim = of(assessments);
    doReturn(version).when(claim).getVersion();

    NtaLogic nta = new NtaLogic(claim, secondary);
    nta = spy(nta);

    return nta;

  }


  @Test
  public void testStep1$2ParenteralIvFeedingCondition() {
    Supplier<Boolean> hasK0510A2 = () -> Boolean.TRUE;
    Supplier<Integer> k0710A2Value = () -> 3;
    Supplier<Integer> k0710B2Value = () -> -1;

    NtaLogic nta = make();
    boolean expected = true;
    boolean actual = nta.step1$2ParenteralIvFeedingCondition(hasK0510A2, k0710A2Value, k0710B2Value)
        .test(new NtaComorbidityRow("", ParenteralIvFeeding.HighIntensity.getItems(), 1));
    Assertions.assertEquals(expected, actual);

    SortedSet<String> items =
        SnfUtils.toOrderedSet(ParenteralIvFeeding.HighIntensity.getItems(), false);
    items.add("xxx");
    expected = false;
    actual = nta.step1$2ParenteralIvFeedingCondition(hasK0510A2, k0710A2Value, k0710B2Value)
        .test(new NtaComorbidityRow("", items, 1));
    Assertions.assertEquals(expected, actual);

    items = SnfUtils.toOrderedSet(ParenteralIvFeeding.HighIntensity.getItems(), false);
    items.remove(items.first());
    expected = false;
    actual = nta.step1$2ParenteralIvFeedingCondition(hasK0510A2, k0710A2Value, k0710B2Value)
        .test(new NtaComorbidityRow("", items, 1));
    Assertions.assertEquals(expected, actual);

    k0710A2Value = () -> 4;
    k0710B2Value = () -> 2;
    items = SnfUtils.toOrderedSet(ParenteralIvFeeding.HighIntensity.getItems(), false);
    expected = false;
    actual = nta.step1$2ParenteralIvFeedingCondition(hasK0510A2, k0710A2Value, k0710B2Value)
        .test(new NtaComorbidityRow("", items, 1));
    Assertions.assertEquals(expected, actual);

    k0710A2Value = () -> 1;
    k0710B2Value = () -> 2;
    items = SnfUtils.toOrderedSet(ParenteralIvFeeding.HighIntensity.getItems(), false);
    expected = false;
    actual = nta.step1$2ParenteralIvFeedingCondition(hasK0510A2, k0710A2Value, k0710B2Value)
        .test(new NtaComorbidityRow("", items, 1));
    Assertions.assertEquals(expected, actual);

    k0710A2Value = () -> 2;
    k0710B2Value = () -> 2;
    items = SnfUtils.toOrderedSet(ParenteralIvFeeding.HighIntensity.getItems(), false);
    expected = false;
    actual = nta.step1$2ParenteralIvFeedingCondition(hasK0510A2, k0710A2Value, k0710B2Value)
        .test(new NtaComorbidityRow("", items, 1));
    Assertions.assertEquals(expected, actual);

    k0710A2Value = () -> 2;
    k0710B2Value = () -> 2;
    items = SnfUtils.toOrderedSet(ParenteralIvFeeding.LowIntensity.getItems(), false);
    expected = true;
    actual = nta.step1$2ParenteralIvFeedingCondition(hasK0510A2, k0710A2Value, k0710B2Value)
        .test(new NtaComorbidityRow("", items, 1));
    Assertions.assertEquals(expected, actual);

    k0710A2Value = () -> 1;
    k0710B2Value = () -> 2;
    items = SnfUtils.toOrderedSet(ParenteralIvFeeding.LowIntensity.getItems(), false);
    expected = false;
    actual = nta.step1$2ParenteralIvFeedingCondition(hasK0510A2, k0710A2Value, k0710B2Value)
        .test(new NtaComorbidityRow("", items, 1));
    Assertions.assertEquals(expected, actual);

    k0710A2Value = () -> 2;
    k0710B2Value = () -> 1;
    items = SnfUtils.toOrderedSet(ParenteralIvFeeding.LowIntensity.getItems(), false);
    expected = false;
    actual = nta.step1$2ParenteralIvFeedingCondition(hasK0510A2, k0710A2Value, k0710B2Value)
        .test(new NtaComorbidityRow("", items, 1));
    Assertions.assertEquals(expected, actual);

    hasK0510A2 = () -> Boolean.FALSE;
    k0710A2Value = () -> 3;
    k0710B2Value = () -> 3;
    items = SnfUtils.toOrderedSet(ParenteralIvFeeding.HighIntensity.getItems(), false);
    expected = false;
    actual = nta.step1$2ParenteralIvFeedingCondition(hasK0510A2, k0710A2Value, k0710B2Value)
        .test(new NtaComorbidityRow("", items, 1));
    Assertions.assertEquals(expected, actual);

    hasK0510A2 = () -> Boolean.FALSE;
    k0710A2Value = () -> 2;
    k0710B2Value = () -> 2;
    items = SnfUtils.toOrderedSet(ParenteralIvFeeding.LowIntensity.getItems(), false);
    expected = false;
    actual = nta.step1$2ParenteralIvFeedingCondition(hasK0510A2, k0710A2Value, k0710B2Value)
        .test(new NtaComorbidityRow("", items, 1));
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testStep2NtaScore() {
    SnfTables.clinicalCategoryMapping.get(null);

    Set<String> secondaryDxNtaCategories =
        SnfUtils.toSet("Cystic Fibrosis", "Lung Transplant Status");// 1, 3
    // Set<String> assessmentNames = SnfUtils.toSet("K0510A2", "K0710A2"); // 7

    NtaLogic nta = make();

    Predicate<NtaComorbidityRow> parenteralIvFeedingFilter = (row) -> false;
    Predicate<NtaComorbidityRow> step1$3AdditionalCommorbidities = (row) -> false;

    int expected = 4;
    int actual = nta.step2NtaScore(secondaryDxNtaCategories, step1$3AdditionalCommorbidities,
        parenteralIvFeedingFilter);
    Assertions.assertEquals(expected, actual);

    secondaryDxNtaCategories = SnfUtils.toSet("Cystic Fibrosis", "Lung Transplant Status");// 1,3
    parenteralIvFeedingFilter =
        NtaLogic.FeedingFilter.apply(ParenteralIvFeeding.HighIntensity.getItems());
    expected = 11;
    actual = nta.step2NtaScore(secondaryDxNtaCategories, step1$3AdditionalCommorbidities,
        parenteralIvFeedingFilter);
    Assertions.assertEquals(expected, actual);

    secondaryDxNtaCategories = SnfUtils.toSet("Cystic Fibrosis", "Lung Transplant Status");// 1,3
    parenteralIvFeedingFilter = (item) -> false;
    expected = 4;
    actual = nta.step2NtaScore(secondaryDxNtaCategories, step1$3AdditionalCommorbidities,
        parenteralIvFeedingFilter);
    Assertions.assertEquals(expected, actual);

    secondaryDxNtaCategories = SnfUtils.toSet("Cystic Fibrosis", "Lung Transplant Status");// 1,3
    step1$3AdditionalCommorbidities =
        (row) -> nta.step1$3AdditionalCommorbidities(SnfUtils.toSet("M1040A", "M1040C"), row); // 1
    parenteralIvFeedingFilter = (item) -> false;
    expected = 5;
    actual = nta.step2NtaScore(secondaryDxNtaCategories, step1$3AdditionalCommorbidities,
        parenteralIvFeedingFilter);
    Assertions.assertEquals(expected, actual);
  }

  @Test
  public void testStep3NtaCaseMixGroup() {
    NtaLogic nta = make();
    for (NtaCmgRow row : SnfTables.ntaCmgTable.values().stream().flatMap(list -> list.stream())
        .collect(Collectors.toSet())) {
      int ntaScoreLow = row.getLowScore();
      int ntaScoreHigh = SnfUtils.nullCheck(row.getHighScore(), 99, row.getHighScore());
      String expected = row.getCmg();
      String actual = nta.step3NtaCaseMixGroup(ntaScoreHigh);
      Assertions.assertEquals(expected, actual);

      expected = row.getCmg();
      actual = nta.step3NtaCaseMixGroup(ntaScoreLow);
      Assertions.assertEquals(expected, actual);

    }

    try {
      nta.step3NtaCaseMixGroup(-1);
      Assertions.fail();
    } catch (Exception ex) {
      Assertions.assertTrue(true);
    }
  }


  @Test
  public void testExec() {
    BiFunction<String, String, SnfDiagnosisCode> mkCode = (value, nc) -> {
      SnfDiagnosisCode code = new SnfDiagnosisCode(value, null, null, nc);
      return code;
    };

    List<SnfDiagnosisCode> secondary = Arrays.asList(mkCode.apply("", "Cystic Fibrosis"),
        mkCode.apply("", "Lung Transplant Status"));
    NtaLogic nta =
        make(Arrays.asList(new Assessment("K0510A2", "", 1), new Assessment("K0710A2", "", 3)),
            secondary);
    String expected = "ND";
    String actual = nta.exec();
    // Assertions.assertEquals(expected, actual);

    nta = make(Arrays.asList(new Assessment("K0510A2", "", 1), new Assessment("K0710A2", "", 3)),
        secondary);
    expected = "ND";
    actual = nta.exec();
    Assertions.assertEquals(expected, actual);
  }
}
